/*
 * Decompiled with CFR 0.152.
 */
package jenkins.telemetry;

import com.google.common.annotations.VisibleForTesting;
import edu.umd.cs.findbugs.annotations.CheckForNull;
import edu.umd.cs.findbugs.annotations.NonNull;
import hudson.Extension;
import hudson.ExtensionList;
import hudson.ExtensionPoint;
import hudson.PluginWrapper;
import hudson.ProxyConfiguration;
import hudson.Util;
import hudson.model.AsyncPeriodicWork;
import hudson.model.TaskListener;
import hudson.model.UsageStatistics;
import hudson.util.VersionNumber;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.LocalDate;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import jenkins.model.Jenkins;
import jenkins.telemetry.Correlator;
import jenkins.util.SystemProperties;
import net.sf.json.JSONObject;
import org.kohsuke.accmod.Restricted;
import org.kohsuke.accmod.restrictions.NoExternalUse;

public abstract class Telemetry
implements ExtensionPoint {
    @Restricted(value={NoExternalUse.class})
    @VisibleForTesting
    static String ENDPOINT = SystemProperties.getString(Telemetry.class.getName() + ".endpoint", "https://uplink.jenkins.io/events");
    private static final Logger LOGGER = Logger.getLogger(Telemetry.class.getName());

    @NonNull
    public String getId() {
        return this.getClass().getName();
    }

    @NonNull
    public abstract String getDisplayName();

    @NonNull
    public abstract LocalDate getStart();

    @NonNull
    public abstract LocalDate getEnd();

    @CheckForNull
    public abstract JSONObject createContent();

    public static ExtensionList<Telemetry> all() {
        return ExtensionList.lookup(Telemetry.class);
    }

    @Restricted(value={NoExternalUse.class})
    public static boolean isAnyTrialActive() {
        return Telemetry.all().stream().anyMatch(Telemetry::isActivePeriod);
    }

    public static boolean isDisabled() {
        if (UsageStatistics.DISABLED) {
            return true;
        }
        Jenkins jenkins = Jenkins.getInstanceOrNull();
        return jenkins == null || !jenkins.isUsageStatisticsCollected();
    }

    public boolean isActivePeriod() {
        LocalDate now = LocalDate.now();
        return now.isAfter(this.getStart()) && now.isBefore(this.getEnd());
    }

    protected final Map<String, String> buildComponentInformation() {
        TreeMap<String, String> components = new TreeMap<String, String>();
        VersionNumber core = Jenkins.getVersion();
        components.put("jenkins-core", core == null ? "" : core.toString());
        for (PluginWrapper plugin : Jenkins.get().pluginManager.getPlugins()) {
            if (!plugin.isActive()) continue;
            components.put(plugin.getShortName(), plugin.getVersion());
        }
        return components;
    }

    @Extension
    public static class TelemetryReporter
    extends AsyncPeriodicWork {
        public TelemetryReporter() {
            super("telemetry collection");
        }

        @Override
        public long getRecurrencePeriod() {
            return TimeUnit.HOURS.toMillis(24L);
        }

        @Override
        protected void execute(TaskListener listener) throws IOException, InterruptedException {
            if (Telemetry.isDisabled()) {
                LOGGER.info("Collection of anonymous usage statistics is disabled, skipping telemetry collection and submission");
                return;
            }
            Telemetry.all().forEach(telemetry -> {
                HttpRequest httpRequest;
                if (telemetry.getStart().isAfter(LocalDate.now())) {
                    LOGGER.config("Skipping telemetry for '" + telemetry.getId() + "' as it is configured to start later");
                    return;
                }
                if (telemetry.getEnd().isBefore(LocalDate.now())) {
                    LOGGER.config("Skipping telemetry for '" + telemetry.getId() + "' as it is configured to end in the past");
                    return;
                }
                JSONObject data = null;
                try {
                    data = telemetry.createContent();
                }
                catch (RuntimeException e) {
                    LOGGER.log(Level.WARNING, "Failed to build telemetry content for: '" + telemetry.getId() + "'", e);
                }
                if (data == null) {
                    LOGGER.log(Level.CONFIG, "Skipping telemetry for '" + telemetry.getId() + "' as it has no data");
                    return;
                }
                JSONObject wrappedData = new JSONObject();
                wrappedData.put("type", (Object)telemetry.getId());
                wrappedData.put("payload", (Object)data);
                String correlationId = ExtensionList.lookupSingleton(Correlator.class).getCorrelationId();
                wrappedData.put("correlator", (Object)Util.getHexOfSHA256DigestOf(correlationId + telemetry.getId()));
                String body = wrappedData.toString();
                if (LOGGER.isLoggable(Level.FINEST)) {
                    LOGGER.finest("Submitting JSON: " + body);
                }
                HttpClient httpClient = ProxyConfiguration.newHttpClient();
                try {
                    URI uri = new URI(ENDPOINT);
                    httpRequest = ProxyConfiguration.newHttpRequestBuilder(uri).headers("Content-Type", "application/json; charset=utf-8").POST(HttpRequest.BodyPublishers.ofString(body)).build();
                }
                catch (IllegalArgumentException | URISyntaxException e) {
                    LOGGER.config("Malformed endpoint URL: " + ENDPOINT + " for telemetry: " + telemetry.getId());
                    return;
                }
                try {
                    HttpResponse<Void> response = httpClient.send(httpRequest, HttpResponse.BodyHandlers.discarding());
                    LOGGER.config("Telemetry submission received response " + response.statusCode() + " for: " + telemetry.getId());
                }
                catch (IOException | InterruptedException e) {
                    LOGGER.log(Level.CONFIG, "Failed to submit telemetry: " + telemetry.getId() + " to: " + ENDPOINT, e);
                }
            });
        }
    }
}

